/*-----------------------------------------------------------------------------
 * sw.c
 * EV2400 firmware
 * 
 * Jennie Kam
 * Version 0.5
 * Last updated: 05/27/09
 * Contains Timer A interrupt handlers that are common to all single wire
 * interfaces.  All single wire setup (H2C, C2H) functions
 * can be found in H2Csetup.c/.h
 * 
 * Peripherials used:
 * 
 * Timer A1, CCR1 (port 1), CCR2 (port 2), from 1MHz ACLK, no divider
 * Timer A1, CCR0 (timeout), from 1MHz ACLK, no divider
 * ---------------------------------------------------------------------------*/

//#include "..\Common\device.h"
#include <intrinsics.h>
//#include "..\Common\main.h"
#include "sw.h"
#include "sdq.h"

/*----------------------------------------------------------------------------+
| Timer A variables
+----------------------------------------------------------------------------*/
unsigned char ISRMode;
unsigned short Xfer;
unsigned char BitCnt;
unsigned int Ticks;
unsigned char delay_done=0, write_done=0, read_done=0, sw_done=0, pulse_done=0;
unsigned char timeout_flag=0;
unsigned char dq_flag;

 /*----------------------------------------------------------------------------+
| Timer A Handlers
+----------------------------------------------------------------------------*/
#pragma vector=TIMERA1_VECTOR
__interrupt void TIMERA1_ISR(void)
{
  // Timer A1, CCR1-2
  switch(__even_in_range(TAIV,4))
  {
    case 0x02: //CCR1
      switch (__even_in_range(ISRMode,18))
      {
        case imWrite :
          if (--BitCnt)                                       // Decr and check
          {
            TACCR1 = Ticks + tCYCH + (tCYCH_DQ * dq_flag);   // Time to cycle completion
            ISRMode = imWriteE;                               // Switch ISR mode
          }
          else
          {
            TACCTL1 = OUTMOD_0 + OUT;                        // Set OUT0, disable int
            write_done=1;   
          }
          break;
        case imWriteE :
          Ticks = TACCR1;                                    // Bit start time stamp
            
          if ((Xfer >>= 1) & 0x01)                            // Process next bit
            TACCR1 = TACCR1 + tHW1 + (tHW1_DQ * dq_flag);   // Output '1'
          else
            TACCR1 = TACCR1 + tHW0 + (tHW0_DQ * dq_flag);   // Output '0'
            
          ISRMode = imWrite;                                  // Switch ISR mode
          break;
        case imRead :
          TACCTL0 = 0;                                       // Disable Time-Out
          TACCR1 += ((tDW0+tDW1) + (tDW0_DQ+tDW1_DQ)*dq_flag) / 2; // Time to middle of bit
          TACCTL1 &= ~CAP;                                   // Compare mode
          ISRMode = imReadE;                                  // Switch ISR mode
          break;
        case imReadE :
          Xfer >>= 1;                                         // Process next bit
          if (TACCTL1 & SCCI)                                // Check Timer_A latch
            Xfer |= 0x8000;
          if (--BitCnt)                                       // Decr and check
          {
            TACCTL1 |= CAP;                                  // Capture mode
            TACCR0 = TAR + tTO + (tTO_DQ * dq_flag);        // Time-Out
            TACCTL0 = CCIE;                                  // Enable Time-Out
            ISRMode = imRead;                                 // Switch ISR mode     
          }
          else
          {
            TACCTL1 = OUTMOD_0 + OUT;                        // Set OUT0, disable int
            read_done=1;   
          }
          break;
        case imDelay :
          TACCTL1 &= ~CCIE;                                  // Disable int
          delay_done = 1;                                     // Return delay done
          break;
        case imSDQwrite:
          TACCTL1 = OUTMOD_0 + OUT;                          // Set OUT0, disable int
          write_done=1; 
        break;
        case imSDQpulse:
          P1DIR &= ~P1_SDQ;                                    // Input to look for presence pulse
          TACCR1 = TACCR1 + tSDQPres;
          ISRMode = imSDQpulseE;
        break;         
      } //switch of ISR Mode 
    break;
    case 0x04: //CCR2
    break; 
  } //switch of TAIV
} //ISR

#pragma vector=TIMERA0_VECTOR
__interrupt void TIMERA0_ISR(void)
{
  //Timer A1, CCR0: used for HDQ timeout
  TACCTL1 = 0;                              // disable capturing on TA.1
  timeout_flag = 1;  
  TACCTL0 = 0;                              // disable timeout interrupt
}
